<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Subscription;
use App\Client;
use App\Product;
use Auth;
use Log;

class SubscriptionController extends Controller
{
	/*=========================================================
	 SubscriptionController 
	 This controller has list of all clients that are on subscription
	 For eg service fee payment which is reoccurring

	 Privileges Used
	 ----------------
	 ACCESS_CLIENT_VIEW
	 ACCESS_SUBSCRIPTION_VIEW - Access to View subscription
	 ACCESS_ADD_SUBSCRIPTION - Access to Add subscription
	 ACCESS_EDIT_SUBSCRIPTION - Access to Edit subscription
	 ACCESS_DELETE_SUBSCRIPTION - Access to Delete subscription
	 ACCESS_CLIENT_SUBSCRIPTION - Access for clients to view their subscriptions
	 ========================================================== */

	 public function __construct()
	 {
	 	$this->middleware('auth');
	 }

	 public function view_client_details($id)
	 {
	 	    if(ConfigController::check_privilege('ACCESS_CLIENT_VIEW'))
	 	    {
	 	    	$client = Client::withTrashed()->find($id);
	 	    	
	 			return view('clients.subscription',compact('id','client'));
	 	    }
	 	    else
	 	    {
	 	        return view('errors.not_authorized');
	 	    }
	 }


	 public function view_client_subscription($id,Request $request)
	 {
 	    if(ConfigController::check_privilege('ACCESS_CLIENT_SUBSCRIPTION'))
 	    {
 			$subscriptions = Subscription::where('client_id',$id)->orderBy('id','desc')->withTrashed()->get();

			if($request->ajax()) 
            {
            	$products = Product::select('id','name','type','amount')->get();
                $view = view('clients.sub_data',compact('subscriptions','products','id'))->render();

                return response()->json(['html'=>$view]);
            }

 	    }
 	    else
 	    {
 			return response()->json(['error'=>'Sorry, you do not have access to this service. Please contact Admin.']);
 	    }
	 }

	 public function load_client_subscription(Request $request)
	 {
 	    if(ConfigController::check_privilege('ACCESS_CLIENT_SUBSCRIPTION'))
 	    {
 			$subscriptions = Subscription::where('client_id',$request->client_id)->orderBy('id','desc')->withTrashed()->get();

			if($request->ajax()) 
            {
            	$products = Product::select('id','name','type','amount')->get();
                $view = view('clients.sub_data_table',compact('subscriptions','products','id'))->render();

                return response()->json(['html'=>$view]);
            }

 	    }
 	    else
 	    {
 			return response()->json(['error'=>'Sorry, you do not have access to this service. Please contact Admin.']);
 	    }	 	
	 }

	public function view_subscription(Request $request)
	{
		if(ConfigController::check_privilege('ACCESS_SUBSCRIPTION_VIEW'))
	    {
			$subscriptions = Subscription::all();

			if($request->ajax()) 
            {
                $view = view('subscriptions.data',compact('subscriptions'))->render();

                return response()->json(['html'=>$view]);
            }

            $clients = Client::select('id','business_name')->get();

			return view('subscriptions.list',compact('subscriptions','clients'));
	    }
	    else
	    {
	        return view('errors.not_authorized');
	    }
	}



	public function add_subscription(Request $request)
	{
	    if(ConfigController::check_privilege('ACCESS_ADD_CLIENT'))
	    {
	    	$this->validate($request,[
	    			'product_id'   	 	  =>		'required',
	    			'type'   		  	  =>		'required',
	    			'amount'   	  		  =>		'required',
	    			'client_id'   	  	  =>		'required'
	    		]);


	    	//Determine Days
	    	switch ($request->type) 
	    	{
	    		case 'one':
	    			$days = 0;
	    			break;
	    		
	    		case 'monthly':
	    			$days = 30;
	    			break;
	    		
	    		case 'quarterly':
	    			$days = 90;
	    			break;
	    		
	    		case 'bi':
	    			$days = 182;
	    			break;
	    		
	    		case 'annually':
	    			$days = 365;
	    			break;
	    	}

			// Check if there is an existing subscriptions available
	    	$next_payment = SubscriptionController::calculate_next_payment($request->client_id,$days);

	    	try
	    	{

	    		$subscription = new Subscription;
				$subscription->product_id    	= $request->product_id;
				$subscription->amount 	   		= $request->amount;
				$subscription->client_id 	   	= $request->client_id;
				$subscription->type 		   	= $request->type;
				$subscription->days 		   	= $days;
				$subscription->next_payment 	= $next_payment;
				$subscription->updated_by    	= Auth::user()->id;
				$subscription->created_by    	= Auth::user()->id;
				$subscription->save();

				return response()->json(['success'=>'Subscription added successfully.']);

	    	}
	    	catch(\Exception $e)
	    	{
	    		Log::ERROR('=============ADD Subscription ERROR ================');
	    		Log::ERROR($e);
	    		Log::ERROR('=============ADD Subscription ERROR END ================');
	    		return response()->json(['error'=>'Sorry, Something went wrong. Please contact Admin.']);
	    	}

	    }
	    else
	    {
			return response()->json(['error'=>'Sorry, you do not have access to this service. Please contact Admin.']);
	    }
	}


	public static function calculate_next_payment($client_id,$days)
	{
		$subscriptions = Subscription::where('client_id',$client_id)
									 ->where('days', '<>','0')
									 ->orderBy('next_payment', 'desc')
									 ->take(1)
									 ->get();

		if(count($subscriptions) > 0)
		{
			foreach ($subscriptions as $subscription) {
				return $subscription->next_payment;
			}
		}
		else
		{
			$today = date("Y-m-d");
			return date('Y-m-d', strtotime($today. ' + '.$days.' day'));
		}
	}



	public function edit_subscription(Request $request)
	{
	    if(ConfigController::check_privilege('ACCESS_EDIT_CLIENT'))
	    {
	    	$this->validate($request,[
	    			'client_id'   	  	=>		'required',
	    			'id'   		  		=>		'required',
	    			'product_id'   	  	=>		'required',
	    			'type'  	 		=>		'required',
	    			'amount'   		  	=>		'required'		
	    		]);

	    	$type = 0;
	    	//Determine Days
	    	switch ($request->type) 
	    	{
	    		case 'one':
	    			$days = 0;
	    			break;
	    		
	    		case 'monthly':
	    			$days = 30;
	    			break;
	    		
	    		case 'quarterly':
	    			$days = 90;
	    			break;
	    		
	    		case 'bi':
	    			$days = 182;
	    			break;
	    		
	    		case 'annually':
	    			$days = 365;
	    			break;
	    	}

			// Check if there is an existing subscriptions available
	    	$next_payment = SubscriptionController::calculate_next_payment($request->client_id,$days);


	    	try
	    	{

	    		$subscription = Subscription::find($request->id);
				$subscription->client_id 		= $request->client_id;
				$subscription->product_id 	= $request->product_id;
				$subscription->type 			= $request->type;
				$subscription->days 		   	= $days;
				$subscription->next_payment 	= $next_payment;
				$subscription->amount 		= $request->amount;
				$subscription->updated_by = Auth::user()->id;
				$subscription->save();

				return response()->json(['success'=>'Subscription edited successfully.']);

	    	}
	    	catch(\Exception $e)
	    	{
	    		Log::ERROR('=============EDIT subscription ERROR ================');
	    		Log::ERROR($e);
	    		Log::ERROR('=============EDIT subscription ERROR END ================');
	    		return response()->json(['error'=>'Sorry, Something went wrong. Please contact Admin.']);
	    	}

	    }
	    else
	    {
			return response()->json(['error'=>'Sorry, you do not have access to this service. Please contact Admin.']);
	    }
	}



	public function delete_subscription(Request $request)
	{
	    if(ConfigController::check_privilege('ACCESS_DELETE_SUBSCRIPTION'))
	    {

	    	try
	    	{

	    		$subscription = Subscription::find($request->id);
				$subscription->updated_by = Auth::user()->id;
				$subscription->save();//update
				$subscription->delete();//delete

				return response()->json(['success'=>'Subscription Deleted successfully.']);

	    	}
	    	catch(\Exception $e)
	    	{
	    		Log::ERROR('=============Delete SUBSCRIPTION ERROR ================');
	    		Log::ERROR($e);
	    		Log::ERROR('=============Delete SUBSCRIPTION ERROR END ================');
	    		return response()->json(['error'=>'Sorry, Something went wrong. Please contact Admin.']);
	    	}

	    }
	    else
	    {
			return response()->json(['error'=>'Sorry, you do not have access to this service. Please contact Admin.']);
	    }
	}


	public function restore_subscription(Request $request)
	{
	    if(ConfigController::check_privilege('ACCESS_RESTORE_SUBSCRIPTION'))
	    {
	    	try
	    	{
	    		
	    		$client = Subscription::withTrashed()->find($request->id);
				$client->updated_by = Auth::user()->id;
				$client->save();//update
				$client->restore();//restore

				return response()->json(['success'=>'Subscription Restored successfully.']);

	    	}
	    	catch(\Exception $e)
	    	{
	    		Log::ERROR('=============Delete SUBSCRIPTION ERROR ================');
	    		Log::ERROR($e);
	    		Log::ERROR('=============Delete SUBSCRIPTION ERROR END ================');
	    		return response()->json(['error'=>'Sorry, Something went wrong. Please contact Admin.']);
	    	}
	
	    }
	    else
	    {
	        return view('errors.not_authorized');
			return response()->json(['unauthorized_access'=>'Sorry, you do not have access to this service. Please contact Admin.']);
	    }
	}


}
